// $Id: CGraphics.cpp,v 1.15 2007/03/07 22:18:51 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CGraphics.hpp"
#ifndef WIN32
#include "../Windowing/CWindowTools.hpp"
using Exponent::GUI::Windowing::CWindowTools;
#endif
using Exponent::GUI::Graphics::CGraphics;

#ifdef WIN32
#include <gdiplus.h>
using namespace Gdiplus;
// Uncomment this next flag to disable GDI+ drawing
//#ifndef DISABLE_GDIPLUS_DRAWING
//#define DISABLE_GDIPLUS_DRAWING
//#endif
#endif

//	===========================================================================
CGraphics::CGraphics()
		 : m_pen(NULL)
		 , m_brush(NULL)
		 , m_nativeImage(NULL)
		 , m_isAlphaBlending(false)
		 , m_textColour(CAlphaColour::CALPHACOLOUR_BLACK)
#ifdef WIN32
		 , m_drawContext(NULL)
		 , m_hasGdiPlus(false)
		 , m_gdiPlusToken(0)

#else
		 , m_theContext(NULL)
#endif
{
	// Create the pen and the brush
	m_pen   = new CPen();
	m_brush = new CBrush();

	// Set the text colour to the default (black i believe ;)
	this->setTextColour();

	// We are not currently alpha blending
	m_isAlphaBlending = false;

#ifdef WIN32
	m_nativeImage = new CNativeImage();
	NULL_POINTER(m_drawContext);
#ifndef DISABLE_GDIPLUS_DRAWING
	// Now we try to attatch to the GDI+ startup process
	GdiplusStartupInput startupInput;
	if (GdiplusStartup(&m_gdiPlusToken, &startupInput, NULL) == Ok)
	{
		m_hasGdiPlus = true;
	}
#endif
#else
	NULL_POINTER(m_nativeImage);
	NULL_POINTER(m_theContext);
#endif
}

//	===========================================================================
CGraphics::~CGraphics()
{
	FREE_POINTER(m_pen);
	FREE_POINTER(m_brush);
#ifdef WIN32
	FREE_POINTER(m_nativeImage);
	NULL_POINTER(m_drawContext);
	if (m_hasGdiPlus)
	{
		GdiplusShutdown(m_gdiPlusToken);
	}
#else
	NULL_POINTER(m_theContext);
#endif
}

#ifdef WIN32
//	===========================================================================
void CGraphics::initialise(SWindowHandle *windowHandle, HDC drawContext, const CDimension &area)
{
	// Init the double buffer
	m_nativeImage->initialise(windowHandle, drawContext, area);
	m_drawContext = m_nativeImage->getGraphicsHandle()->m_drawContext;

	// Init the pen and the brush
	m_pen->initialise(1, m_drawContext, CAlphaColour::CALPHACOLOUR_BLACK);
	m_brush->initialise( m_drawContext, CAlphaColour::CALPHACOLOUR_WHITE);

	// Intiialised the clipping rectangle
	m_clip.initialise(m_nativeImage, CRect(0, 0, area.getWidth(), area.getHeight()));

	// Store the root area
	m_rootArea = area;

//#ifndef DISABLE_GDIPLUS_DRAWING
//	// Now we try to attatch to the GDI+ startup process
//	GdiplusStartupInput startupInput;
//	if (GdiplusStartup(&m_gdiPlusToken, &startupInput, NULL) == Ok)
//	{
//		m_hasGdiPlus = true;
//	}
//#endif
}
#else
//	===========================================================================
void CGraphics::initialise(SWindowHandle *windowHandle, CGContextRef drawContext, const CDimension &area)
{
	// Init the pen and the brush
	m_pen->initialise(drawContext, 1, CAlphaColour::CALPHACOLOUR_BLACK);
	m_brush->initialise(drawContext, CAlphaColour::CALPHACOLOUR_WHITE);

	// Intiialised the clipping rectangle
	m_clip.initialise(NULL, CRect(0, 0, area.getWidth(), area.getHeight()), drawContext);

	// Store the context
	m_theContext = drawContext;
	CGContextRetain(m_theContext);

	// Store the root area...
	m_rootArea = area;
}
#endif

//	===========================================================================
void CGraphics::uninitialise()
{
	// Uninitialise the clip, pen and brush
	m_clip.uninitialise();
	m_pen->uninitialise();
	m_brush->uninitialise();

#ifdef WIN32
	//if (m_hasGdiPlus)
	//{
	//	GdiplusShutdown(m_gdiPlusToken);
	//}
	m_nativeImage->uninitialise();
	NULL_POINTER(m_drawContext);
#else
	CGContextRelease(m_theContext);
	NULL_POINTER(m_theContext);
#endif
}

//	===========================================================================
void CGraphics::beginAlphaDrawing(const double alpha)
{
	// Cannot be nested
	if (m_isAlphaBlending)
	{
		return;
	}
#ifdef WIN32

	// Store the alpha
	m_alphaGraphics.m_alphaValue = alpha;

	// Create the native image to draw to
	m_alphaGraphics.m_alphaImage = new CNativeImage();

	// Initialise the new image
	m_alphaGraphics.m_alphaImage->initialise(m_nativeImage->getWindowHandle(), m_nativeImage->getParentContext(), m_rootArea);
	m_nativeImage->copyTo(m_alphaGraphics.m_alphaImage);

	// Store new context
	m_drawContext = m_alphaGraphics.m_alphaImage->getGraphicsHandle()->m_drawContext;

	// Store the old brush and pen
	m_alphaGraphics.m_oldBrush = m_brush;
	m_alphaGraphics.m_oldPen   = m_pen;

	// Create and setup new pen and brush
	CPen *alphaPen     = new CPen();
	CBrush *alphaBrush = new CBrush();
	alphaPen->initialise(m_pen->getWidth(), m_drawContext, m_pen->getColour());
	alphaBrush->initialise(m_drawContext, m_brush->getColour());

	// Store them
	m_brush = alphaBrush;
	m_pen   = alphaPen;

	/*
	// Intiialised the clipping rectangle
	CRect theClip;
	m_clip.getCurrentClipRegion(theClip);
	m_alphaClip.initialise(m_nativeImage, theClip);
	*/
#else
	CGContextSetAlpha(m_theContext, (float)alpha);
#endif
	// We are now alpha blending
	m_isAlphaBlending = true;
}

//	===========================================================================
void CGraphics::endAlphaDrawing(const CRect &drawTo, const CRect &offset)
{
	// Cannot be nested
	if (!m_isAlphaBlending)
	{
		return;
	}
#ifdef WIN32

	// Delete the alpha pen and brush
	m_pen->uninitialise();
	m_brush->uninitialise();
	FREE_POINTER(m_brush);
	FREE_POINTER(m_pen);

	// Store the originals
	m_brush = m_alphaGraphics.m_oldBrush;
	m_pen   = m_alphaGraphics.m_oldPen;

	// Store destination and offset
	CRect destination = drawTo;
	destination.offset(m_graphicsOffset);

	CRect theOffset = offset;
	theOffset.offset(m_graphicsOffset);

	// Create the blend struction
	BLENDFUNCTION myBlendFunction;
	myBlendFunction.BlendFlags			= 0;
	myBlendFunction.BlendOp			    = AC_SRC_OVER;
	myBlendFunction.AlphaFormat			= 0;
	myBlendFunction.SourceConstantAlpha = (BYTE)(255.0 * m_alphaGraphics.m_alphaValue);

	// Alpha blend between the two panels
	AlphaBlend(m_nativeImage->getGraphicsHandle()->m_drawContext, destination.getLeft(), destination.getTop(), destination.getWidth(), destination.getHeight(),
			   m_alphaGraphics.m_alphaImage->getGraphicsHandle()->m_drawContext,  theOffset.getLeft(), theOffset.getTop(), theOffset.getWidth(), theOffset.getHeight(), myBlendFunction);

	// Release all the contexts
	m_drawContext = m_nativeImage->getGraphicsHandle()->m_drawContext;
	m_alphaGraphics.m_alphaImage->uninitialise();
	FREE_POINTER(m_alphaGraphics.m_alphaImage);
	//m_alphaClip.uninitialise();
#else
	CGContextSetAlpha(m_theContext, 1.f);
#endif
	// We are no longer alpha blending...
	m_isAlphaBlending = false;
}

//	===========================================================================
void CGraphics::drawPixel(const CPoint &point)
{
	CPoint myPoint = point;
	myPoint.offset(m_graphicsOffset);
#ifdef WIN32
	SetPixelV(m_drawContext, myPoint.getXPosition(), myPoint.getYPosition(), m_pen->getColour().getAsColourRef());
#else
	// Stupid mac, they cant make it easy can they.... :(
	// To draw a single pixel on the mac, we need to offset by 0.5f, then draw a line of length 1
	CGContextBeginPath(m_theContext);

	// Offset the postion for quartz
	myPoint.setYPosition(m_rootArea.getHeight() - myPoint.getYPosition());

	// Store as floating point, remembering to move it to the center of the pixel
	const float x = myPoint.getXPosition();
	const float y = myPoint.getYPosition();

	// Add the point
	CGContextMoveToPoint(   m_theContext, x, y);
	CGContextAddLineToPoint(m_theContext, x + 1.f, y + 1.f);

	// draw the line
	CGContextStrokePath(m_theContext);
#endif
}

//	===========================================================================
void CGraphics::drawAntiAliasedPixel(const CPoint &point, const long alpha)
{
	CPoint myPoint = point;
	myPoint.offset(m_graphicsOffset);
#ifdef WIN32

	// First we need to get the colour of the pixel to be overwritten
	COLORREF colour = GetPixel(m_drawContext, myPoint.getXPosition(), myPoint.getYPosition());
	CColour backgroundColour(colour);
	CAlphaColour blend;
	CAlphaColour current = m_pen->getColour();

	// Now we do the actual alpha blending
	CAlphaColour::alphaBlend(blend, backgroundColour, current, (unsigned long)alpha);

	// Set the pen colour
	m_pen->setColour(blend);

	// Finally we need to draw the blended colour back
	SetPixelV(m_drawContext, myPoint.getXPosition(), myPoint.getYPosition(), blend.getAsColourRef());

	// Resotre the old colour
	m_pen->setColour(current);

#else
	this->drawPixel(point);	// Already is anti aliased on the mac
#endif
}

//	===========================================================================
void CGraphics::moveTo(const CPoint &point)
{
	CPoint myPoint = point;
	myPoint.offset(m_graphicsOffset);
#ifdef WIN32
	MoveToEx(m_drawContext, myPoint.getXPosition(), myPoint.getYPosition(), NULL);
#else
	CGContextBeginPath(m_theContext);
	myPoint.setYPosition(m_rootArea.getHeight() - myPoint.getYPosition());
	CGContextMoveToPoint(m_theContext, (float)myPoint.getXPosition(), (float)myPoint.getYPosition());
#endif
}

//	===========================================================================
void CGraphics::drawLine(const CPoint &start, const CPoint &end)
{
	// Store the position
	CPoint myStart = start;
	CPoint myEnd   = end;

	// Offset the position
	myStart.offset(m_graphicsOffset);
	myEnd.offset(m_graphicsOffset);
#ifdef WIN32
	MoveToEx(m_drawContext, myStart.getXPosition(), myStart.getYPosition(), NULL);
	LineTo(m_drawContext,   myEnd.getXPosition(),   myEnd.getYPosition());
#else
	CGContextBeginPath(m_theContext);
	myStart.setYPosition(m_rootArea.getHeight() - myStart.getYPosition());
	myEnd.setYPosition(  m_rootArea.getHeight() - myEnd.getYPosition());

	// Add the points
	CGContextMoveToPoint(m_theContext, (float)myStart.getXPosition(), (float)myStart.getYPosition());
	CGContextAddLineToPoint(m_theContext, (float)myEnd.getXPosition(), (float)myEnd.getYPosition());

	// Draw the line
	CGContextStrokePath(m_theContext);
#endif
}

//	===========================================================================
void CGraphics::drawLine(const CPoint &toThisPoint)
{
	CPoint myPoint = toThisPoint;
	myPoint.offset(m_graphicsOffset);
#ifdef WIN32
	LineTo(m_drawContext, myPoint.getXPosition(), myPoint.getYPosition());
#else
	CGContextBeginPath(m_theContext);
	myPoint.setYPosition(m_rootArea.getHeight() - myPoint.getYPosition());
	CGContextAddLineToPoint(m_theContext, (float)myPoint.getXPosition(), (float)myPoint.getYPosition());
	CGContextStrokePath(m_theContext);
#endif
}

#ifdef WIN32
//	===========================================================================
void CGraphics::drawWUAALine(const CPoint &start, const CPoint &end)
{
	// Implements the Xiaolin Wu antialiased line algorithm...
	// References:
	// http://en.wikipedia.org/wiki/Xiaolin_Wu's_line_algorithm
	// http://www.suchit-tiwari.org/download/aawu.c

	CPoint startPoint = start;
	CPoint endPoint   = end;
	startPoint.offset(m_graphicsOffset);
	endPoint.offset(m_graphicsOffset);

	int x0 = (int)startPoint.getXPosition();
	int y0 = (int)startPoint.getYPosition();
	int x1 = (int)endPoint.getXPosition();
	int y1 = (int)endPoint.getYPosition();
	
	// Ensure that the line goes top to bottom
    if (y0 > y1)
    {
        int store = y0; 
        y0        = y1; 
        y1        = store;
        store     = x0; 
        x0        = x1; 
        x1        = store;
    }

	// Store the pen colour
	COLORREF penColour = m_pen->getColour().getAsColourRef();

	// The initial pixel is always intersected fully and is there fore drawn properly...
	SetPixelV(m_drawContext, x0, y0, penColour);

	// Determine the direction and x increment...
	int xdir, deltax = x1 - x0;
    if( deltax >= 0 )
    {
        xdir = 1;
    }
    else
    {
        xdir   = -1;
        deltax = 0 - deltax; 
    }

	// From http://www.suchit-tiwari.org/download/aawu.c : 'Special-case horizontal, vertical, and diagonal lines, which require no weighting because they go right through the center of every pixel' 
    int deltay = y1 - y0;
    if (deltay == 0)
    {
        // Horizontal line 
        while (deltax-- != 0)
        {
            x0 += xdir;
            SetPixelV(m_drawContext, x0, y0, penColour);
        }
        return;
    }
    if (deltax == 0)
    {
        // Vertical line 
        do
        {
            y0++;
			SetPixelV(m_drawContext, x0, y0, penColour);
        } while (--deltay != 0);
        return;
    }
    
    if (deltax == deltay)
    {
        // Diagonal line 
        do
        {
            x0 += xdir;
            y0++;
			SetPixelV(m_drawContext, x0, y0, penColour);
        } 
		while (--deltay != 0);
        return;
    }

	// If we have got to here we are neither horizontal, vertical or diagonal

	// Some temporary variables for use during the loop
	unsigned short errorAdjacency;
    unsigned short errorAcumulatorTemp;
	unsigned short pixelWeighting;
    unsigned short errorAcumulator = 0;  /* initialize the line error accumulator to 0 */
    
	// Get the individual pen colour components
    BYTE rl      = GetRValue(penColour);
    BYTE gl      = GetGValue(penColour);
    BYTE bl      = GetBValue(penColour);
    double grayl = rl * 0.299 + gl * 0.587 + bl * 0.114;
    
    // Is this an X-major or Y-major line?
    if (deltay > deltax)
    {
    	// Y-major line; calculate 16-bit fixed-point fractional part of a pixel that X advances each time Y advances 1 pixel, truncating the result so that we won't overrun the endpoint along the X axis 
        errorAdjacency = (unsigned short)(((unsigned long) deltax << 16) / (unsigned long) deltay);
        
        // Draw all pixels other than the first and last 
        while (--deltay) 
        {
			// remember currrent accumulated error 
            errorAcumulatorTemp = errorAcumulator; 

			// calculate error for next pixel 
            errorAcumulator += errorAdjacency;      

			// Check if The error accumulator turned over, so advance the X coord 
            if (errorAcumulator <= errorAcumulatorTemp) 
            {
                x0 += xdir;
            }

			// Y-major, so always advance Y 
            y0++; 
                  
			// The IntensityBits most significant bits of errorAcumulator give us the intensity weighting for this pixel, and the complement of the weighting for the paired pixel 
            pixelWeighting = errorAcumulator >> 8;
            
			// Get the background colours and the blend level
            COLORREF backgroundColour = GetPixel(m_drawContext, x0, y0);
            BYTE rb                   = GetRValue(backgroundColour);
            BYTE gb                   = GetGValue(backgroundColour);
            BYTE bb                   = GetBValue(backgroundColour);
            double grayb              = rb * 0.299 + gb * 0.587 + bb * 0.114;
            
			// Compute the actual alpha blend
            BYTE rr = ( rb > rl ? ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting: (pixelWeighting ^ 255)) ) / 255.0 * ( rb - rl ) + rl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting:(pixelWeighting ^ 255)) )/ 255.0 * ( rl - rb ) + rb ) ) );
            BYTE gr = ( gb > gl ? ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting: (pixelWeighting ^ 255)) ) / 255.0 * ( gb - gl ) + gl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting:(pixelWeighting ^ 255)) ) / 255.0 * ( gl - gb ) + gb ) ) );
            BYTE br = ( bb > bl ? ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting: (pixelWeighting ^ 255)) ) / 255.0 * ( bb - bl ) + bl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting:(pixelWeighting ^ 255)) ) / 255.0 * ( bl - bb ) + bb ) ) );
            SetPixelV(m_drawContext, x0, y0, RGB(rr, gr, br));
            
			// Get the background colours and the blend level for the edge pixels
            backgroundColour = GetPixel(m_drawContext, x0 + xdir, y0);
            rb				 = GetRValue(backgroundColour);
            gb				 = GetGValue(backgroundColour);
            bb				 = GetBValue(backgroundColour);
            grayb			 = rb * 0.299 + gb * 0.587 + bb * 0.114;
            
			// Compute the alpha blend for the edges
            rr = ( rb > rl ? ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255): pixelWeighting) ) / 255.0 * ( rb - rl ) + rl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255):pixelWeighting) ) / 255.0 * ( rl - rb ) + rb ) ) );
            gr = ( gb > gl ? ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255): pixelWeighting) ) / 255.0 * ( gb - gl ) + gl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255):pixelWeighting) ) / 255.0 * ( gl - gb ) + gb ) ) );
            br = ( bb > bl ? ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255): pixelWeighting) ) / 255.0 * ( bb - bl ) + bl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255):pixelWeighting) ) / 255.0 * ( bl - bb ) + bb ) ) );
            SetPixelV(m_drawContext, x0 + xdir, y0, RGB(rr, gr, br));
        }
        
        // Draw the final pixel, which is always exactly intersected by the line and so needs no weighting
        SetPixelV(m_drawContext, x1, y1, penColour);
        return;
    }

	// It's an X-major line; calculate 16-bit fixed-point fractional part of a pixel that Y advances each time X advances 1 pixel, truncating the result to avoid overrunning the endpoint along the X axis */
    errorAdjacency = (unsigned short)(((unsigned long) deltay << 16) / (unsigned long) deltax);

    // Draw all pixels other than the first and last 
    while (--deltax) 
    {
		// remember currrent accumulated error
        errorAcumulatorTemp = errorAcumulator;   

		 // calculate error for next pixel 
		errorAcumulator += errorAdjacency;     

		// Check if The error accumulator turned over, so advance the Y coord
        if (errorAcumulator <= errorAcumulatorTemp) 
        {
            y0++;
        }

		// X-major, so always advance X 
        x0 += xdir; 
        
		// The IntensityBits most significant bits of errorAcumulator give us the intensity weighting for this pixel, and the complement of the weighting for the paired pixel 
        pixelWeighting = errorAcumulator >> 8;
        
		// Get the background colour and blend
        COLORREF backgroundColour = GetPixel(m_drawContext, x0, y0);
        BYTE rb					  = GetRValue(backgroundColour);
        BYTE gb					  = GetGValue(backgroundColour);
        BYTE bb					  = GetBValue(backgroundColour);
        double grayb			  = rb * 0.299 + gb * 0.587 + bb * 0.114;
        
		// Do the alpha blend and draw
        BYTE rr = ( rb > rl ? ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting: (pixelWeighting ^ 255)) ) / 255.0 * ( rb - rl ) + rl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting:(pixelWeighting ^ 255)) ) / 255.0 * ( rl - rb ) + rb ) ) );
        BYTE gr = ( gb > gl ? ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting: (pixelWeighting ^ 255)) ) / 255.0 * ( gb - gl ) + gl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting:(pixelWeighting ^ 255)) ) / 255.0 * ( gl - gb ) + gb ) ) );
        BYTE br = ( bb > bl ? ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting: (pixelWeighting ^ 255)) ) / 255.0 * ( bb - bl ) + bl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?pixelWeighting:(pixelWeighting ^ 255)) ) / 255.0 * ( bl - bb ) + bb ) ) );
        SetPixelV(m_drawContext, x0, y0, RGB(rr, gr, br));
        
		// Get background side colours
		backgroundColour = GetPixel(m_drawContext, x0, y0 + 1);
        rb				 = GetRValue(backgroundColour);
        gb				 = GetGValue(backgroundColour);
        bb				 = GetBValue(backgroundColour);
        grayb			 = rb * 0.299 + gb * 0.587 + bb * 0.114;
        
		// Do side blend and draw
        rr = ( rb > rl ? ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255): pixelWeighting) ) / 255.0 * ( rb - rl ) + rl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255):pixelWeighting) ) / 255.0 * ( rl - rb ) + rb ) ) );
        gr = ( gb > gl ? ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255): pixelWeighting) ) / 255.0 * ( gb - gl ) + gl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255):pixelWeighting) ) / 255.0 * ( gl - gb ) + gb ) ) );
        br = ( bb > bl ? ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255): pixelWeighting) ) / 255.0 * ( bb - bl ) + bl ) ) : ( ( BYTE )( ( ( double )( grayl<grayb?(pixelWeighting ^ 255):pixelWeighting) ) / 255.0 * ( bl - bb ) + bb ) ) );
        SetPixelV(m_drawContext, x0, y0 + 1, RGB(rr, gr, br));
    }
        
    // Draw the final pixel, which is always exactly intersected by the line and so needs no weighting
    SetPixelV(m_drawContext, x1, y1, penColour);
}
#endif

//	===========================================================================
void CGraphics::drawAntiAliasedLine(const CPoint &start, const CPoint &end)
{
#ifdef WIN32
	if (m_hasGdiPlus)
	{
		Gdiplus::Graphics graphics(m_drawContext);
		Pen thePen(Color((unsigned char)m_pen->getColour().getAlpha(), (unsigned char)m_pen->getColour().getRed(), (unsigned char)m_pen->getColour().getGreen(), (unsigned char)m_pen->getColour().getBlue()), (Gdiplus::REAL)m_pen->getWidth());

		// Now setup the points to draw
		CPoint startPoint = start;
		CPoint endPoint   = end;
		startPoint.offset(m_graphicsOffset);
		endPoint.offset(m_graphicsOffset);

		// Convert to GDI+ points
		Point theStart;
		Point theEnd;
		theStart.X = startPoint.getXPosition();
		theStart.Y = startPoint.getYPosition();
		theEnd.X   = endPoint.getXPosition();
		theEnd.Y   = endPoint.getYPosition();

		SmoothingMode mode = graphics.GetSmoothingMode();
		graphics.SetSmoothingMode(SmoothingModeHighQuality);
		graphics.DrawLine(&thePen, theStart, theEnd);
		graphics.SetSmoothingMode(mode);
	}
	else
	{
		this->drawWUAALine(start, end);
	}
#else
	this->drawLine(start, end);
#endif
}

//	===========================================================================
void CGraphics::drawRectangle(const CRect &area)
{
	CRect myArea = area;
	myArea.offset(m_graphicsOffset);
#ifdef WIN32
	CAlphaColour oldColour = m_brush->getColour();
	m_brush->setColourToNullBrush();
	Rectangle(m_drawContext, myArea.getLeft(), myArea.getTop(), myArea.getRight(), myArea.getBottom());
	m_brush->setColour(oldColour);
#else
	if (m_theContext == NULL)
	{
		return;
	}
	
	//myArea.inset(1);
	CGContextBeginPath(m_theContext);
	CGRect rect;
	rect.origin.x    = myArea.getLeft() + 0.5f;
	rect.origin.y    = (m_rootArea.getHeight() - myArea.getTop() - myArea.getHeight()) + 0.5f;
	rect.size.width  = myArea.getWidth() - 1.0f;
	rect.size.height = myArea.getHeight() - 1.0f;

	CGContextAddRect(m_theContext, rect);
	CGContextStrokePath(m_theContext);
#endif
}

//	===========================================================================
void CGraphics::drawRoundedRectangle(const CRect &rect, const CDimension &roundingEffect)
{
	CRect myArea = rect;
	myArea.offset(m_graphicsOffset);
#ifdef WIN32
	CAlphaColour oldColour = m_brush->getColour();
	m_brush->setColourToNullBrush();
	RoundRect(m_drawContext, myArea.getLeft(), myArea.getTop(), myArea.getRight(), myArea.getBottom(), roundingEffect.getWidth(), roundingEffect.getHeight());
	m_brush->setColour(oldColour);
#else
	const float left   = myArea.getLeft() + 0.5f;
	const float right  = myArea.getRight() + 0.5f;
	const float top    = m_rootArea.getHeight() - myArea.getTop() + 0.5f;
	const float bottom = m_rootArea.getHeight() - myArea.getBottom() + 0.5f;
	const float radius = (float)roundingEffect.getWidth();

	CGContextBeginPath(m_theContext);
	CGContextAddArc(m_theContext, left  - radius, top - radius,    radius, CMathTools::CMATH_PI_FLOAT,				CMathTools::CMATH_HALF_PI_FLOAT,		 1);
	CGContextAddArc(m_theContext, right + radius, top - radius,    radius, CMathTools::CMATH_HALF_PI_FLOAT,			0.f,									 1);
	CGContextAddArc(m_theContext, right + radius, bottom + radius, radius, 0.f,										CMathTools::CMATH_PI_FLOAT * 3.f / 2.f,  1);
	CGContextAddArc(m_theContext, left  - radius, bottom + radius, radius, CMathTools::CMATH_PI_FLOAT * 3.f / 2.f,	CMathTools::CMATH_PI_FLOAT,				 1);
	CGContextClosePath(m_theContext);
	CGContextStrokePath(m_theContext);
#endif
}

//	===========================================================================
void CGraphics::drawPolygon(const CPolygon &polygon)
{
#ifdef WIN32
	CPolygon myPolygon = polygon;
	CPoint *point = myPolygon.getPoint(0);
	this->moveTo(*point);
	for (long i = 0; i < myPolygon.getNumberOfPoints() - 1; i++)
	{
		point = myPolygon.getPoint(i + 1);
		this->drawLine(*point);
	}
#else
	// Check its not a straight line
	if (polygon.getNumberOfPoints() > 2)
	{
		CPolygon myPolygon = polygon;
		CPoint *point1;
		CPoint *point2;
		for (long i = 0; i < myPolygon.getNumberOfPoints() - 1; i++)
		{
			point1 = myPolygon.getPoint(i);
			point2 = myPolygon.getPoint(i + 1);
			if (point1 && point2)
			{
				this->drawLine(*point1, *point2);
			}
		}
	}
#endif
}

//	===========================================================================
void CGraphics::drawAntiAliasedPolygon(const CPolygon &polygon)
{
	// Check its not a straight line
	if (polygon.getNumberOfPoints() > 2)
	{
		CPolygon myPolygon = polygon;
		CPoint *point1;
		CPoint *point2;
		for (long i = 0; i < myPolygon.getNumberOfPoints() - 1; i++)
		{
			point1 = myPolygon.getPoint(i);
			point2 = myPolygon.getPoint(i + 1);
			if (point1 && point2)
			{
				this->drawAntiAliasedLine(*point1, *point2);
			}
		}
	}
}

//	===========================================================================
void CGraphics::drawEllipse(const CRect &area)
{
#ifdef WIN32
	this->drawArc(CRect(area.getLeft(), area.getTop(), area.getWidth(), area.getHeight()), CPoint(0, 0), CPoint(0, 0));
#else
	CRect myRect = area;
	myRect.offset(m_graphicsOffset);
	const long midPoint = myRect.getLeft() + myRect.getWidth() / 2;
	const long top      = m_rootArea.getHeight() - (myRect.getTop() + (myRect.getHeight() / 2));
	CGContextAddArc(m_theContext, midPoint + 1, top, myRect.getWidth() / 2, 0, 2 * CMathTools::CMATH_2PI_FLOAT, 1);
	CGContextStrokePath(m_theContext);
#endif
}

//	===========================================================================
void CGraphics::drawArc(const CRect &area, const CPoint &startPos, const CPoint &endPos, const bool clockwise)
{
	CRect myArea   = area;
	CPoint myStart = startPos;
	CPoint myEnd   = endPos;

	myArea.offset(m_graphicsOffset);
	myStart.offset(m_graphicsOffset);
	myEnd.offset(m_graphicsOffset);

#ifdef WIN32
	SetArcDirection(m_drawContext, (clockwise) ? AD_CLOCKWISE : AD_COUNTERCLOCKWISE);
	Arc(m_drawContext, myArea.getLeft(), myArea.getTop(), myArea.getRight(), myArea.getBottom(), myStart.getXPosition(), myStart.getYPosition(), myEnd.getXPosition(), myEnd.getYPosition());
#else
	// Do something here....
#endif
}

//	===========================================================================
void CGraphics::fillRectangle(const CRect &rect)
{
	CRect myRect = rect;
	myRect.offset(m_graphicsOffset);
#ifdef WIN32
	if (m_hasGdiPlus)
	{
		Gdiplus::Graphics graphics(m_drawContext);
		SmoothingMode mode = graphics.GetSmoothingMode();
		graphics.SetSmoothingMode(SmoothingModeHighQuality);
		SolidBrush brush(Color((unsigned char)m_brush->getColour().getAlpha(), (unsigned char)m_brush->getColour().getRed(), (unsigned char)m_brush->getColour().getGreen(), (unsigned char)m_brush->getColour().getBlue()));
		graphics.FillRectangle(&brush, myRect.getLeft(), myRect.getTop(), myRect.getWidth(), myRect.getHeight());
		graphics.SetSmoothingMode(mode);
	}
	else
	{
		CAlphaColour oldColour = m_pen->getColour();
		m_pen->setPenToNullPen();
		RECT area = myRect.getAsRect();
		FillRect(m_drawContext, &area, m_brush->getBrush());
		m_pen->setColour(oldColour);
	}
#else
//	myRect.inset(1);
	CGContextBeginPath(m_theContext);

	// Create the fill area
	const long left   = myRect.getLeft();
	const long right  = myRect.getRight();
	const long top    = (m_rootArea.getHeight() - myRect.getTop());
	const long bottom = (m_rootArea.getHeight() - myRect.getBottom());

	// Add the points to the path
	CGContextMoveToPoint(   m_theContext, left,  top);
	CGContextAddLineToPoint(m_theContext, right, top);
	CGContextAddLineToPoint(m_theContext, right, bottom);
	CGContextAddLineToPoint(m_theContext, left,  bottom);
	CGContextAddLineToPoint(m_theContext, left,  top);

	// Fill!
	CGContextSetAlpha(m_theContext, ((float)m_brush->getColour().getAlpha()) / 255.0);
	CGContextFillPath(m_theContext);
	CGContextSetAlpha(m_theContext, 1.f);
#endif
}

//	===========================================================================
void CGraphics::fillRoundedRectangle(const CRect &rect, const CDimension &roundingEffect)
{
	CRect myRect = rect;
	myRect.offset(m_graphicsOffset);
#ifdef WIN32
	if (m_hasGdiPlus)
	{
		const long radius		= roundingEffect.getWidth();
		const long doubleRadius = radius * 2;
		GraphicsPath graphicsPath;
		graphicsPath.AddLine(	myRect.getLeft()  + radius, myRect.getTop(), myRect.getLeft() + myRect.getWidth() - doubleRadius, myRect.getTop());
		graphicsPath.AddArc(	myRect.getRight() - doubleRadius, myRect.getTop(), doubleRadius, doubleRadius, 270, 90);
		graphicsPath.AddLine(	myRect.getRight(), myRect.getTop() + radius, myRect.getLeft() + myRect.getWidth(), myRect.getTop() + myRect.getHeight() - doubleRadius);
		graphicsPath.AddArc(	myRect.getRight() - doubleRadius, myRect.getBottom() - doubleRadius, doubleRadius, doubleRadius, 0, 90);
		graphicsPath.AddLine(	myRect.getRight() - doubleRadius, myRect.getBottom(), myRect.getLeft() + radius, myRect.getTop() + myRect.getHeight());
		graphicsPath.AddArc(	myRect.getLeft(), myRect.getBottom() - doubleRadius, doubleRadius, doubleRadius, 90, 90);
		graphicsPath.AddLine(	myRect.getLeft(), myRect.getBottom() - doubleRadius, myRect.getLeft(), myRect.getTop() + radius);
		graphicsPath.AddArc(	myRect.getLeft(), myRect.getTop(), doubleRadius, doubleRadius, 180, 90);
		graphicsPath.CloseFigure();
		
		SolidBrush brush(Color((unsigned char)m_brush->getColour().getAlpha(), (unsigned char)m_brush->getColour().getRed(), (unsigned char)m_brush->getColour().getGreen(), (unsigned char)m_brush->getColour().getBlue()));
		Gdiplus::Graphics graphics(m_drawContext);
		SmoothingMode mode = graphics.GetSmoothingMode();
		graphics.SetSmoothingMode(SmoothingModeHighQuality);
		graphics.FillPath(&brush, &graphicsPath);
		graphics.SetSmoothingMode(mode);
	}
	else
	{
		CAlphaColour oldColour = m_pen->getColour();
		m_pen->setPenToNullPen();
		RoundRect(m_drawContext, myRect.getLeft(), myRect.getTop(), myRect.getRight(), myRect.getBottom(), roundingEffect.getWidth(), roundingEffect.getHeight());
		m_pen->setColour(oldColour);
	}
#else
	const float left   = myRect.getLeft() + 0.5f;
	const float right  = myRect.getRight() + 0.5f;
	const float top    = m_rootArea.getHeight() - myRect.getTop() + 0.5f;
	const float bottom = m_rootArea.getHeight() - myRect.getBottom() + 0.5f;
	const float radius = (float)roundingEffect.getWidth();

	CGContextSetAlpha(m_theContext, ((float)m_brush->getColour().getAlpha()) / 255.0);
	CGContextBeginPath(m_theContext);
	CGContextAddArc(m_theContext, left  + radius, top - radius,    radius, CMathTools::CMATH_PI_FLOAT,				CMathTools::CMATH_HALF_PI_FLOAT,		 1);
	CGContextAddArc(m_theContext, right - radius, top - radius,    radius, CMathTools::CMATH_HALF_PI_FLOAT,			0.f,									 1);
	CGContextAddArc(m_theContext, right - radius, bottom + radius, radius, 0.f,										CMathTools::CMATH_PI_FLOAT * 3.f / 2.f,  1);
	CGContextAddArc(m_theContext, left  + radius, bottom + radius, radius, CMathTools::CMATH_PI_FLOAT * 3.f / 2.f,	CMathTools::CMATH_PI_FLOAT,				 1);
	CGContextClosePath(m_theContext);
	CGContextFillPath(m_theContext);
	CGContextSetAlpha(m_theContext, 1.f);
#endif
}

//	===========================================================================
void CGraphics::fillEllipse(const CRect &rect)
{
	CRect myRect = rect;
	myRect.offset(m_graphicsOffset);
#ifdef WIN32
	if (m_hasGdiPlus)
	{
		Gdiplus::Graphics graphics(m_drawContext);
		SmoothingMode mode = graphics.GetSmoothingMode();
		graphics.SetSmoothingMode(SmoothingModeHighQuality);
		SolidBrush brush(Color((unsigned char)m_brush->getColour().getAlpha(), (unsigned char)m_brush->getColour().getRed(), (unsigned char)m_brush->getColour().getGreen(), (unsigned char)m_brush->getColour().getBlue()));
		graphics.FillEllipse(&brush, myRect.getLeft(), myRect.getTop(), myRect.getWidth(), myRect.getHeight());
		graphics.SetSmoothingMode(mode);
	}
	else
	{
		CAlphaColour oldColour = m_pen->getColour();
		m_pen->setPenToNullPen();
		Ellipse(m_drawContext, myRect.getLeft(), myRect.getTop(), myRect.getRight(), myRect.getBottom());
		m_pen->setColour(oldColour);
	}
#else
	CGContextSetAlpha(m_theContext, ((float)m_brush->getColour().getAlpha()) / 255.0);
	const long midPoint = myRect.getLeft() + myRect.getWidth() / 2;
	const long top      = m_rootArea.getHeight() - (myRect.getTop() + (myRect.getHeight() / 2));
	CGContextAddArc(m_theContext, midPoint + 1, top, myRect.getWidth() / 2, 0, 2 * CMathTools::CMATH_2PI_FLOAT, 1);
	CGContextFillPath(m_theContext);
	CGContextSetAlpha(m_theContext, 1.f);
#endif
}

//	===========================================================================
void CGraphics::fillPolygon(const CPolygon &polygon)
{
	CPolygon myPolygon = polygon;
	myPolygon.offset(m_graphicsOffset);
#ifdef WIN32

	if (m_hasGdiPlus)
	{
		Gdiplus::Graphics graphics(m_drawContext);
		SmoothingMode mode = graphics.GetSmoothingMode();
		graphics.SetSmoothingMode(SmoothingModeHighQuality);
		SolidBrush brush(Color((unsigned char)m_brush->getColour().getAlpha(), (unsigned char)m_brush->getColour().getRed(), (unsigned char)m_brush->getColour().getGreen(), (unsigned char)m_brush->getColour().getBlue()));

		Point *points = new Point[myPolygon.getNumberOfPoints()];
		for (long i = 0; i < myPolygon.getNumberOfPoints(); i++)
		{
			CPoint *point = myPolygon.getPoint(i);
			if (point)
			{
				points[i].X = point->getXPosition();
				points[i].Y = point->getYPosition();
			}
		}

		graphics.FillPolygon(&brush, points, myPolygon.getNumberOfPoints());
		graphics.SetSmoothingMode(mode);

	}
	else
	{
		CAlphaColour oldColour = m_pen->getColour();
		m_pen->setPenToNullPen();
		POINT *points = new POINT[myPolygon.getNumberOfPoints()];
		for (long i = 0; i < myPolygon.getNumberOfPoints(); i++)
		{
			CPoint *point = myPolygon.getPoint(i);
			if (point)
			{
				points[i].x = point->getXPosition();
				points[i].y = point->getYPosition();
			}
		}
		Polygon(m_drawContext, points, myPolygon.getNumberOfPoints());
		FREE_ARRAY_POINTER(points);
		m_pen->setColour(oldColour);
	}
#else
	if (polygon.getNumberOfPoints() < 2)
	{
		return;
	}
	// Begin the path
	CGContextBeginPath(m_theContext);

	// Store the point array
	TCountedPointerArray<CPoint> *points = myPolygon.getMutablePointArray();

	// Get the origin point
	CPoint *origin = points->elementAtIndex(0);

	// Move to the origin point
	CGContextMoveToPoint(m_theContext, origin->getXPosition() + 0.5f, (m_rootArea.getHeight() - origin->getYPosition()) + 0.5f);

	// Loop through and add the extra points
	for (long i = 1; i < myPolygon.getNumberOfPoints(); i++)
	{
		CPoint *nextPoint = points->elementAtIndex(i);
		if (nextPoint)
		{
			CGContextAddLineToPoint(m_theContext, nextPoint->getXPosition() + 0.5f, (m_rootArea.getHeight() - nextPoint->getYPosition()) + 0.5f);
		}
	}

	// Close the path
	CGContextAddLineToPoint(m_theContext, origin->getXPosition() + 0.5f, (m_rootArea.getHeight() - origin->getYPosition()) + 0.5f);

	// Fill!
	CGContextSetAlpha(m_theContext, ((float)m_brush->getColour().getAlpha()) / 255.0);
	CGContextFillPath(m_theContext);
	CGContextSetAlpha(m_theContext, 1.f);
#endif
}

//	===========================================================================
void CGraphics::drawFilledBorderedRectangle(const CRect &area)
{
	// First we draw the fill area
	this->fillRectangle(area);

	// Now we want to draw the border
	this->drawRectangle(area);
}

//	===========================================================================
void CGraphics::drawGradientFilledRectangle(const CRect &area)
{
	// We area going to gradient fill the colour, ranged from pen colour at the edges, to white in the middle then back to pen colour
	// Across each height increment, we blend to the new colour, then draw a line a single pixel high
	// White (target) is 255,255,255
	// The pen colour will always be less than this...
	// Each line is one pixel high
	
	const long totalHeight = area.getHeight();
	const long height      = totalHeight / 2;
	
	// We want to compute the offset from each to the size of the area
	const CAlphaColour penColour = this->getMutablePen()->getColour();
	const long offsets[] = {(255 - penColour.getRed()) / height, (255 - penColour.getGreen()) / height, (255 - penColour.getBlue()) / height};

	CPoint start(0, 0);
	CPoint end(area.getWidth(), 0);

	// Store the colour we are fading
	CAlphaColour theColour = penColour;

	// Loop and draw
	for (long i = 0; i < totalHeight; i++)
	{
		// Set the colour of the pen
		this->getMutablePen()->setColour(theColour);

		// Draw the line
		this->drawLine(start, end);

		// Mov the points
		start.offset(0, 1);
		end.offset(0, 1);

		if (i >= height)
		{
			theColour.setRed(theColour.getRed()     - offsets[0]);
			theColour.setGreen(theColour.getGreen() - offsets[1]);
			theColour.setBlue(theColour.getBlue()   - offsets[2]);
		}
		else
		{
			theColour.setRed(theColour.getRed()     + offsets[0]);
			theColour.setGreen(theColour.getGreen() + offsets[1]);
			theColour.setBlue(theColour.getBlue()   + offsets[2]);
		}
	}

	// Restore the old colour
	this->getMutablePen()->setColour(penColour);
}

#include <iostream>
using namespace std;

//	===========================================================================
void CGraphics::drawImage(IImage *image, const CRect &drawTo, const CRect &offset)
{
	if (image && image->hasImageLoaded() && image->imageHasAlpha())
	{
		this->drawImageWithAlpha(image, drawTo, offset, 1.0);
	}
	else if (image && image->hasImageLoaded())
	{
		CRect destination = drawTo;
		destination.offset(m_graphicsOffset);
		
#ifdef WIN32
		HDC sourceDC      = CreateCompatibleDC(m_drawContext);
		HGDIOBJ oldObj    = SelectObject(sourceDC, image->getImageHandle());

		if (image->getImageSize().getWidth()  == destination.getWidth() &&
			image->getImageSize().getHeight() == destination.getHeight())
		{
			BitBlt(m_drawContext, 
			       destination.getLeft(), 
			       destination.getTop(), 
			       destination.getWidth(), 
			       destination.getHeight(), 
				   sourceDC, 
				   offset.getLeft(), 
				   offset.getTop(), 
				   SRCCOPY);
		}
		else
		{
            StretchBlt(m_drawContext, 
			           destination.getLeft(), 
			           destination.getTop(), 
			           destination.getWidth(), 
			           destination.getHeight(), 
				       sourceDC, 
				       offset.getLeft(), 
				       offset.getTop(), 
					   offset.getWidth(), 
					   offset.getHeight(), 
				       SRCCOPY);
		}

		// Reselect the original dc
		SelectObject(sourceDC, oldObj);
		DeleteDC(sourceDC);
#else

		cerr << "We are drawing an image" << endl;
		
		// Compute the image position
		CGRect clipRect;
		clipRect.origin.x    = destination.getLeft();
		clipRect.origin.y    = destination.getTop();
		clipRect.size.width  = destination.getWidth();
		clipRect.size.height = destination.getHeight();
		clipRect.origin.y    = m_rootArea.getHeight() - (clipRect.origin.y + destination.getHeight());

		cerr << "Setup the clipping area" << endl;
		// Clip to the area
		CGContextSaveGState(m_theContext);
		CGContextClipToRect(m_theContext, clipRect);

		// Now draw the image
		if (image->getImageSize().getWidth() != offset.getWidth() || image->getImageSize().getHeight() != offset.getHeight())
		{
			CGRect destinationRect      = clipRect;
			destinationRect.size.width  = image->getImageSize().getWidth();
			destinationRect.size.height = image->getImageSize().getHeight();
			destinationRect.origin.x    = destination.getLeft() - offset.getLeft();
			destinationRect.origin.y    = destination.getTop()  - offset.getTop() + image->getImageSize().getHeight() - destination.getHeight();
			destinationRect.origin.y    = m_rootArea.getHeight() - (destinationRect.origin.y + destination.getHeight());
			CGContextDrawImage(m_theContext, destinationRect, image->getImageHandle());
		}
		else
		{
			CGContextDrawImage(m_theContext, clipRect, image->getImageHandle());
		}

		// Restore the clipped rectangle
		CGContextRestoreGState(m_theContext);
#endif
	}
}

//	===========================================================================
void CGraphics::drawImageWithAlpha(IImage *image, const CRect &drawTo, const CRect &offset, const double alpha)
{
	if (image && image->hasImageLoaded() && image->imageHasAlpha())
	{
		CRect destination = drawTo;
		destination.offset(m_graphicsOffset);
	#ifdef WIN32
		// Create the blend struction
		BLENDFUNCTION myBlendFunction;
		myBlendFunction.BlendFlags			= 0;
		myBlendFunction.BlendOp			    = AC_SRC_OVER;
		myBlendFunction.AlphaFormat			= AC_SRC_ALPHA;
		myBlendFunction.SourceConstantAlpha = (BYTE)(255.0 * alpha);

		// Create the source and select the image in to it
		HDC destinationDC = m_drawContext;
		HDC sourceDC      = CreateCompatibleDC(destinationDC);
		HGDIOBJ oldObj    = SelectObject(sourceDC, image->getImageHandle());

		// Alpha blend between the two panels
		AlphaBlend(destinationDC, destination.getLeft(), destination.getTop(), destination.getWidth(), destination.getHeight(),
				   sourceDC, offset.getLeft(), offset.getTop(), offset.getWidth(), offset.getHeight(), myBlendFunction);

		// Reselect the original dc
		SelectObject(sourceDC, oldObj);
		DeleteDC(sourceDC);
	#else
		// Compute the image position
		CGRect clipRect;
		clipRect.origin.x    = destination.getLeft();
		clipRect.origin.y    = destination.getTop();
		clipRect.size.width  = destination.getWidth();
		clipRect.size.height = destination.getHeight();
		clipRect.origin.y    = m_rootArea.getHeight() - (clipRect.origin.y + destination.getHeight());

		// Clip to the area
		CGContextSaveGState(m_theContext);
		CGContextClipToRect(m_theContext, clipRect);

		// Now draw the image
		if (image->getImageSize().getWidth() != offset.getWidth() || image->getImageSize().getHeight() != offset.getHeight())
		{
			CGRect destinationRect      = clipRect;
			destinationRect.size.width  = image->getImageSize().getWidth();
			destinationRect.size.height = image->getImageSize().getHeight();
			destinationRect.origin.x    = destination.getLeft() - offset.getLeft();
			destinationRect.origin.y    = destination.getTop()  - offset.getTop() + image->getImageSize().getHeight() - destination.getHeight();
			destinationRect.origin.y    = m_rootArea.getHeight() - (destinationRect.origin.y + destination.getHeight());
			CGContextDrawImage(m_theContext, destinationRect, image->getImageHandle());
		}
		else
		{
			CGContextDrawImage(m_theContext, clipRect, image->getImageHandle());
		}

		// Restore the clipped rectangle
		CGContextRestoreGState(m_theContext);
	#endif
	}
	else if (image && image->hasImageLoaded())
	{
		this->drawImage(image, drawTo, offset);
	}
}

//	===========================================================================
void CGraphics::drawText(const CString &text, const CRect &area, const CFont *theFont, const ETextAlignment drawOptions)
{
	// Store the area
	CRect theArea = area;
	theArea.offset(m_graphicsOffset);

	// Store the font to use
	const CFont *font = (theFont) ? theFont : CFont::CFONT_SYSTEM_FONT;

	// If the font is still null, we're boned..
	if (font == NULL) 
	{
		return;
	}

#ifdef WIN32
	if (m_hasGdiPlus)
	{
		Gdiplus::Graphics graphics(m_drawContext);
		SmoothingMode mode     = graphics.GetSmoothingMode();
		TextRenderingHint hint = graphics.GetTextRenderingHint();
		graphics.SetSmoothingMode(SmoothingModeHighQuality);

		// Set to clear type, if not supported use anti aliasing...
		if (graphics.SetTextRenderingHint(TextRenderingHintClearTypeGridFit) != Ok)
		{
			graphics.SetTextRenderingHint(TextRenderingHintAntiAlias);
		}
		SolidBrush brush(Color((unsigned char)m_textColour.getAlpha(), (unsigned char)m_textColour.getRed(), (unsigned char)m_textColour.getGreen(), (unsigned char)m_textColour.getBlue()));

		// Convert the string to WCHAR
		const size_t length = strlen(text.getString()) + 1;
		wchar_t *theText    = new wchar_t[length];
		memset(theText, 0, length * sizeof(wchar_t));
		MultiByteToWideChar(CP_ACP, NULL, text.getString(), -1, theText, (int)length);

		// Construct the rectangle
		Gdiplus::RectF theRect((float)theArea.getLeft(), (float)theArea.getTop(), (float)theArea.getWidth(), (float)theArea.getHeight());
		Gdiplus::Font theFont(m_drawContext, font->getFont()); 
		StringFormat format(0);
		format.SetFormatFlags(StringFormatFlagsLineLimit);
		format.SetTrimming(StringTrimmingNone);
		
		// Setup the drawing options
		switch(drawOptions)
		{
			case ETextAlignment::e_leftTopAlign:	
				format.SetAlignment(StringAlignmentNear);	
				format.SetLineAlignment(StringAlignmentNear);	
			break;
			case ETextAlignment::e_centerTopAlign:	
				format.SetAlignment(StringAlignmentCenter);	
				format.SetLineAlignment(StringAlignmentNear);	
			break;
			case ETextAlignment::e_rightTopAlign:	
				format.SetAlignment(StringAlignmentFar);	
				format.SetLineAlignment(StringAlignmentNear);	
			break;
			case ETextAlignment::e_leftCenterAlign:
			case ETextAlignment::e_defaultLabelText:
				format.SetAlignment(StringAlignmentNear);	
				format.SetLineAlignment(StringAlignmentCenter);	
			break;
			case ETextAlignment::e_centerCenterAlign:
			case ETextAlignment::e_defaultSingleLine:
				format.SetAlignment(StringAlignmentCenter);	
				format.SetLineAlignment(StringAlignmentCenter);	
			break;
			case ETextAlignment::e_rightCenterAlign:
				format.SetAlignment(StringAlignmentFar);	
				format.SetLineAlignment(StringAlignmentCenter);	
			break;
			case ETextAlignment::e_leftBottomAlign:
				format.SetAlignment(StringAlignmentNear);	
				format.SetLineAlignment(StringAlignmentFar);	
			break;			
			case ETextAlignment::e_centerBottomAlign:
				format.SetAlignment(StringAlignmentCenter);	
				format.SetLineAlignment(StringAlignmentFar);	
			break;					
			case ETextAlignment::e_rightBottomAlign:
				format.SetAlignment(StringAlignmentFar);	
				format.SetLineAlignment(StringAlignmentFar);	
			break;
		}

		graphics.DrawString(theText, -1, &theFont, theRect, &format, &brush);

		// Now delete all memory and release our setup
		FREE_ARRAY_POINTER(theText);
		graphics.SetSmoothingMode(mode);
		graphics.SetTextRenderingHint(hint);
	}
	else
	{
		UINT theDrawOptions;

		switch(drawOptions)
		{
			case e_leftTopAlign:		theDrawOptions = DT_LEFT   | DT_TOP;							break;
			case e_centerTopAlign:		theDrawOptions = DT_CENTER | DT_TOP;							break;
			case e_rightTopAlign:		theDrawOptions = DT_RIGHT  | DT_TOP;							break;
			case e_leftCenterAlign:		theDrawOptions = DT_LEFT   | DT_VCENTER | DT_SINGLELINE;		break;
			case e_defaultLabelText:	theDrawOptions = DT_LEFT   | DT_VCENTER | DT_SINGLELINE;		break;
			case e_centerCenterAlign:	theDrawOptions = DT_CENTER | DT_VCENTER | DT_SINGLELINE;		break;
			case e_defaultSingleLine:	theDrawOptions = DT_CENTER | DT_VCENTER | DT_SINGLELINE;		break;
			case e_rightCenterAlign:	theDrawOptions = DT_RIGHT  | DT_VCENTER | DT_SINGLELINE;		break;
			case e_leftBottomAlign:		theDrawOptions = DT_LEFT   | DT_BOTTOM  | DT_SINGLELINE;		break;					
			case e_centerBottomAlign:	theDrawOptions = DT_CENTER | DT_BOTTOM  | DT_SINGLELINE;		break;					
			case e_rightBottomAlign:	theDrawOptions = DT_RIGHT  | DT_BOTTOM  | DT_SINGLELINE;		break;
		}

		// Load the font
		HGDIOBJ old = SelectObject(m_drawContext, font->getFont());

		// Set the colours to draw with
		SetTextColor(m_drawContext, m_textColour.getAsColourRef());
		SetBkMode(m_drawContext, TRANSPARENT);

		// The are to draw to
		RECT rect = theArea.getAsRect();

		// Draw the text
		DrawText(m_drawContext, text.getString(), text.getNumberOfCharacters() - 1, &rect, theDrawOptions);

		// Restore the previous font
		SelectObject(m_drawContext, old);
	}
#else
	/*
		Warning : This function is a horrible nasty hackish affair
		Because there is not auto align when drawing text on mac, we have
		to compute it ourselves
		This makes life somewhat difficult and this code may not work for
		all font types / size in all text formats.
		Best to just try it and see what happens ;)
	*/

	// Setup the clip area
	CGRect clipRect;
	clipRect.origin.x    = theArea.getLeft();
	clipRect.origin.y    = theArea.getTop();
	clipRect.size.width  = theArea.getWidth();
	clipRect.size.height = theArea.getHeight();
	clipRect.origin.y    = m_rootArea.getHeight() - (clipRect.origin.y + theArea.getHeight());

	// Clip to the area
	CGContextSaveGState(m_theContext);
	CGContextClipToRect(m_theContext, clipRect);

	// Set the text colour
	CGContextSetRGBFillColor(m_theContext, m_textColour.getRed() / 255.f, m_textColour.getGreen() / 255.f, m_textColour.getBlue() / 255.f, 1.f);

	// Set the font
	CGContextSelectFont(m_theContext, font->getFontName().getString(), (float)font->getHeight(), kCGEncodingMacRoman);


	// First we compute the dimensions of the string, this gives us the width and height of the string
	CDimension dimension;
	CWindowTools::computeStringArea(font, text, dimension);
	
	// Now, based on the size of the string, we can apply the drawing options
	const float width    = dimension.getWidth();
	const float vCenter  = (float)theArea.getCentralYPosition();
	const float hCenter  = (float)theArea.getCentralXPosition();
	const float fontSize = dimension.getHeight();

	// Set position based upon drawing options
	switch(drawOptions)
	{
		case e_leftTopAlign:
			{
				const float x = theArea.getLeft() + 0.5f;
				const float y = clipRect.origin.y - 1.5f + (theArea.getHeight() - fontSize);
				CGContextShowTextAtPoint(m_theContext, x, y, text.getString(), text.getNumberOfCharacters() - 1);
			}
		break;
		case e_centerTopAlign:
			{
				const float x = hCenter - (width * 0.5f);
				const float y = clipRect.origin.y - 1.5f + (theArea.getHeight() - fontSize);
				CGContextShowTextAtPoint(m_theContext, x, y, text.getString(), text.getNumberOfCharacters() - 1);
			}
		break;
		case e_rightTopAlign:
			{
				const float x = theArea.getRight() - width - 0.5f;
				const float y = clipRect.origin.y - 1.5f + (theArea.getHeight() - fontSize);
				CGContextShowTextAtPoint(m_theContext, x, y, text.getString(), text.getNumberOfCharacters() - 1);
			}
		break;
		case e_leftCenterAlign:
		case e_defaultLabelText:
			{
				const float x = theArea.getLeft() + 0.5f;
				const float y = m_rootArea.getHeight() - (vCenter + (fontSize * 0.5f)) + 1.5f;
				CGContextShowTextAtPoint(m_theContext, x, y, text.getString(), text.getNumberOfCharacters() - 1);	
			}			
		break;
		case e_centerCenterAlign:
		case e_defaultSingleLine:
			{
				const float x = hCenter - (width * 0.5f);
				const float y = m_rootArea.getHeight() - (vCenter + (fontSize * 0.5f)) + 1.5f;
				CGContextShowTextAtPoint(m_theContext, x, y, text.getString(), text.getNumberOfCharacters() - 1);	
			}		
		break;
		case e_rightCenterAlign:
			{
				const float x = theArea.getRight() - width - 0.5f;
				const float y = m_rootArea.getHeight() - (vCenter + (fontSize * 0.5f)) + 1.5f;
				CGContextShowTextAtPoint(m_theContext, x, y, text.getString(), text.getNumberOfCharacters() - 1);	
			}		
		break;
		case e_leftBottomAlign:
			{ 
				const float x = theArea.getLeft() + 0.5f;
				const float y = m_rootArea.getHeight() - theArea.getBottom() + 0.5f;
				CGContextShowTextAtPoint(m_theContext, x, y, text.getString(), text.getNumberOfCharacters() - 1);
			}	
		break;
		case e_centerBottomAlign:
			{ 
				const float x = hCenter - (width * 0.5f);
				const float y = m_rootArea.getHeight() - theArea.getBottom() + 0.5f;
				CGContextShowTextAtPoint(m_theContext, x, y, text.getString(), text.getNumberOfCharacters() - 1);
			}	
		break;
		case e_rightBottomAlign:
			{ 
				const float x = theArea.getRight() - width - 0.5f;
				const float y = m_rootArea.getHeight() - theArea.getBottom() + 0.5f;
				CGContextShowTextAtPoint(m_theContext, x, y, text.getString(), text.getNumberOfCharacters() - 1);
			}		
		break;
	}

	// Restore the clipped rectangle
	CGContextRestoreGState(m_theContext);
#endif
}

//	===========================================================================
CPen *CGraphics::getMutablePen() const
{
	return m_pen;
}

//	===========================================================================
const CPen *CGraphics::getPen() const
{
	return m_pen;
}

//	===========================================================================
CBrush *CGraphics::getMutableBrush() const
{
	return m_brush;
}

//	===========================================================================
const CBrush *CGraphics::getBrush() const
{
	return m_brush;
}

//	===========================================================================
CClip &CGraphics::getClippingRegion()
{
#ifdef WIN32
	if (m_isAlphaBlending)
	{
		return m_clip;//m_alphaClip;
	}
	else
	{
		return m_clip;
	}
#else
	return m_clip;
#endif
}
